/*
 * Generate a single wirelist file from the database
 */
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>

#include "db.h"
#include "libfma.h"
#include "lf_fabric.h"

static void
usage()
{
  fprintf(stderr, "This is used to see the exact path a packet takes through the network\n");
  fprintf(stderr, "given a starting host and a sequence of route bytes.\n");
  fprintf(stderr, "This is run on a node which has filesystem access to the database files.\n");
  fprintf(stderr, "Since this program does not modify the database, it can be run at any time.\n\n");

  fprintf(stderr, "Usage: \n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -f <hostname> - the host from which to start\n");
  fprintf(stderr, "  -n <nic_id> - which NIC to use\n");
  fprintf(stderr, "  -p <port> - which port on the NIC to use\n");
  fprintf(stderr, "  -g - give extra-gory details about internal links\n");
  fprintf(stderr, "  -R <fms_run> - specify FMS_RUN directory\n");
  fprintf(stderr, "  -N <fms_db_name> - specify database name\n");
  fprintf(stderr, "  -V - print version\n");
  fprintf(stderr, "  -- <route_info> - specify the route to follow\n\n");
  fprintf(stderr, "Example usage: fm_walkroute -f host1 -- bd bf 90 91\n");

  exit(1);
}

union lf_node *
get_next_node(
  lf_node_t *this_node,
  int this_exit_port,
  int *next_entry_port)
{
  lf_node_t *next_node;

  switch (this_node->ln_type) {
  case LF_NODE_NIC_XCVR:
  case LF_NODE_LC_XCVR:
    next_node = LF_XCVR(this_node)->ports[this_exit_port];
    if (next_node != NULL) {
      *next_entry_port = LF_XCVR(this_node)->rports[this_exit_port];
    }
    break;

  case LF_NODE_XBAR:
    next_node = LF_XBAR(this_node)->phys_ports[this_exit_port];
    if (next_node != NULL) {
      *next_entry_port = LF_XBAR(this_node)->phys_rports[this_exit_port];
    }
    break;

  case LF_NODE_NIC:
    next_node = LF_XBAR(this_node)->phys_ports[this_exit_port];
    if (next_node != NULL) {
      *next_entry_port = LF_XBAR(this_node)->phys_rports[this_exit_port];
    }
    break;
  default:
    fprintf(stderr, "Bad node type %d\n", this_node->ln_type);
    next_node = NULL;
    break;
  }

  return next_node;
}

void
print_node_desc(
  lf_node_t *np,
  int port)
{
  struct lf_linecard *lp;

  switch (np->ln_type) {
    case LF_NODE_LC_XCVR:
      lp = LF_XCVR(np)->p.linecard;
      printf("%s, slot %d, port %d",
	     lp->enclosure->name,
	     lf_slot_display_no(lp),
	     lp->xcvr_labels[LF_XCVR(np)->port]);
      break;

    case LF_NODE_XBAR:
      lp = LF_XBAR(np)->linecard;

      printf("%s, slot %d, xbar %d, port %d",
	     lp->enclosure->name,
	     lf_slot_display_no(lp),
	     LF_XBAR(np)->xbar_no,
	     port);
      break;

    case LF_NODE_NIC_XCVR:
      {
	lf_nic_t *nicp;
	lf_host_t *hp;

	nicp = LF_XCVR(np)->p.nic;
	hp = nicp->host;

	printf("host %s, nic %d, rail %d",
	       hp->hostname,
	       nicp->host_nic_id,
	       LF_XCVR(np)->rports[port+LF_XCVR(np)->num_conns]);
	break;
      }

    case LF_NODE_NIC:
      break;
  }
}




void
walk_route(
  struct lf_fabric *fp,
  char *starthost,
  int nic_id,
  int ifc,
  signed char *route,
  int rlen,
  int gory)
{
  struct lf_host *hp;
  union lf_node *this_node;
  union lf_node *next_node;
  int this_exit_port;
  int next_entry_port;
  int ri;
  int ni;

  hp = lf_find_host_by_name(fp, starthost);
  if (hp == NULL) {
    fprintf(stderr, "Cannot find host %s in fabric database\n", starthost);
    exit(1);
  }

  /* get to nic xcvr for this host */
  for (ni=0; ni < hp->num_nics; ++ni) {
    if(hp->nics[ni]->host_nic_id == nic_id) {
      break;
    }
  } 
  if (ni >= hp->num_nics) {
    fprintf(stderr, "No such NIC ID (%d) in %s\n", nic_id, starthost);
    exit(1);
  }

  if (ifc >= hp->nics[ni]->num_ports) {
    fprintf(stderr, "Invalid interface %d, max is %d\n", ifc,
	    hp->nics[ni]->num_ports-1);
    exit(1);
  }

  this_node = hp->nics[ni]->phys_ports[ifc];
  this_exit_port = 0;

  ri = 0;		/* route index */

  while (this_node != NULL) {

    next_node = get_next_node(this_node, this_exit_port, &next_entry_port);


    if (gory ||
	(this_node->ln_type != LF_NODE_XBAR
	 && (next_node == NULL || next_node->ln_type != LF_NODE_XBAR))) {


      print_node_desc(this_node, this_exit_port);
      printf(" - ");

      if (next_node == NULL) {
	printf("nothing\n");
	break;
      }

      print_node_desc(next_node, next_entry_port);

      if (gory) {
	if (this_node->ln_type == LF_NODE_XBAR
	    || next_node->ln_type == LF_NODE_XBAR) {
	  if (this_node->ln_type == LF_NODE_XBAR
	      && next_node->ln_type == LF_NODE_XBAR) {
	    printf(" [BACKPLANE]");
	  } else {
	    printf(" [XCVR-XBAR]");
	  }
	} else {
	  printf(" [FIBER]");
	} 
      }
      printf("\n");
    }

    /* find the next place to go */
    this_node = next_node;
    if (this_node == NULL) {
      printf("No further connections\n");
      break;
    } else if (this_node->ln_type == LF_NODE_LC_XCVR) {
      int np;

      /* calculate exit port for a transceiver */
      np = LF_XCVR(this_node)->num_conns;
      this_exit_port = (next_entry_port + np) % (2*np);

    } else if (this_node->ln_type == LF_NODE_NIC
	       || this_node->ln_type == LF_NODE_NIC_XCVR) {

      /* if we reach a NIC, we are done... */
      break;

    } else {

      /* consume a routing byte to get the next */
      if (ri >= rlen) {
	printf("out of routing bytes before reaching another NIC\n");
	break;
      }

      if (gory) printf("\t[route byte %d]\n", route[ri]);
      this_exit_port = next_entry_port + route[ri];
      if (this_exit_port < 0
	  || this_exit_port >= LF_XBAR(this_node)->num_ports) {
	printf("route byte %d generates out of range port\n", route[ri]);
	exit(1);
      }
      ++ri;
    }
  }
}

int
ascii_route_to_delta(
  char *route)
{
  int r;
  char *cp;
  int nondec;

  nondec=0;
  cp = route;
  if (*cp == '-') ++cp;
  
  while (*cp != '\0') {
    if (!isdigit(*cp)) {
	nondec=1;
	break;
    }
    ++cp;
  }

  if (*route == '-' && nondec) {
    fprintf(stderr, "Bad route byte: %s\n", route);
    exit(1);
  }

  if (!nondec) {
    r = atoi(route);
    if (r < 80) return r;
  }

  sscanf(route, "%x", &r);
  r &= 0x3f;
  if (r & 0x20) {
    r |= (~0 & ~0x3f);		/* sign extend */
  }
  return r;
}


int
main(
  int argc,
  char **argv)
{
  int c;
  extern char *optarg;
  extern int optind;
  char *db_name;
  char *fms_run;
  char *starthost;
  int nic_slot;
  struct lf_fabric *fp;
  signed char route[512];
  int rlen;
  int i;
  int gory;
  int ifc;

  lf_init();

  /* defaults */
  starthost = NULL;
  gory = 0;
  fms_run = NULL;
  db_name = NULL;
  nic_slot = 0;
  ifc = 0;

  /* command line args */
  while ((c = getopt(argc, argv, "H:R:N:hf:p:gn:V")) != EOF) switch (c) {
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  case 'h':
    usage();
    break;
  case 'H':
    fprintf(stderr,
	"Please note: -H has been replaced with -R and is deprecated\n");
    /* FALLSTHROUGH */

  case 'R':
    fms_run = optarg;
    break;
  case 'N':
    db_name = optarg;
    break;
  case 'f':
    starthost = optarg;
    break;
  case 'n':
    nic_slot = atoi(optarg);
    break;
  case 'p':
    ifc = atoi(optarg);
    break;
  case 'g':
    gory = 1;
    break;
  }

  /* set dirs - doesn't care if they are NULL */
  lf_simple_set_fabric_dirs(fms_run, db_name);

  /* need a starting host */
  if (starthost == NULL) {
    fprintf(stderr, "Need a starting host.\n");
    exit(1);
  }

  rlen = 0;
  while (optind < argc) {
    route[rlen++] = ascii_route_to_delta(argv[optind++]);
  }
  printf("Walking route:");
  for (i=0; i<rlen; ++i) {
    printf(" %d", route[i]);
  }
  printf(" from host %s, nic %d, ifc %d\n", starthost, nic_slot, ifc);

  fp = lf_simple_load_fabric();
  if (fp == NULL) {
    fprintf(stderr, "Error loading fabric.\n");
    exit(1);
  }

  walk_route(fp, starthost, nic_slot, ifc, route, rlen, gory);

  exit(0);
}
